# these are the command line steps to deploy TechIDManager.AzureAD by Ruffian Software, Inc.
# All right resevered 
# copyright 2025 Ruffian Software, Inc.
#

param (
#    [switch]$Logout,
	[switch]$GCC
)


$ErrorActionPreference = "Stop"

# Be aware that some of these steps will register resource providers in the azure tenant and can take up to 45 minutes 
# to complete some steps the first time they are run in a tenant.  

# Ensure the Azure CLI is installed and logged in
if (-not (Get-Command "az" -ErrorAction SilentlyContinue)) {
    Write-Error "Azure CLI is not installed or not found in the PATH."
    return
}


# being logged into multiple accounts with legacy TechIDManager installations seems to confuse users, so we just make sure they are only logged into the subscription that they want to install in. 
#if ($Logout) {
	try {
		echo 'Logging out of all connected EntraID/AzureAD tenants'
		$subscripts = (az account list | ConvertFrom-Json)
		while ($subscripts.Count -gt 0) 
		{
 			$sub_name = $subscripts[0].user.name
 			echo "Logging out of : $sub_name "
 			az account set --subscription $subscripts[0].id
 			az logout
 			$subscripts = (az account list | ConvertFrom-Json)
		}
 
	} catch {
		Write-Error 'Problem logging out...trying to install TechIDManager.EntraID anyway...'
	}
#}


$government = 'N'
$basegraph = 'https://graph.microsoft.com'

if ($GCC) {
	az cloud set --name AzureUSGovernment		
	$government = 'Y'
	$basegraph = 'https://graph.microsoft.us'
	echo 'Using GCC/GCC-High Azure'
} else {
	az cloud set --name AzureCloud
	echo 'Using Commercial Azure'
}


try {
	echo 'Azure CLI version'
	az version
	echo 'Waiting on login to Azure'
	az login
} catch {
	Write-Error 'Azure CLI appears to not be installed (or some other error happened, like you need a subscription attached to the tenant, or you did not login sucessfully)' -ErrorAction Stop
}			


$NewInstall = 'Y'
# Function app and storage account names must be unique.
$suffix = Get-Date -Format "MMddHHmm"
$tidregion = 'eastus'
$DomainGuid = ''
$found_something_to_do = 'no'

# define some paths where we save stuff
$PathRSNoVersion = "$HOME\RuffianSoftware\TechIDAgent.EntraID"
$PathRSVersion = "$PathRSNoVersion\6.147"

#make the path where we are saving stuff
New-Item -ItemType Directory -Force -Path $PathRSVersion 


# Check for existing TechIDManager.Azure install by looking for a matching resourceGroupName in each subscription
try {

	$subscripts = (az account list | ConvertFrom-Json)
	foreach ($sub in $subscripts) 
	{
		echo "Checking for TechIDMangaer.EntraID installation in subscription : $sub"
		az account set --subscription $sub.id

		$RGList = (az group list --query "[?starts_with(name, 'techidmgr') && ends_with(name, 'rg')]" | ConvertFrom-Json)
		
		foreach ($RGCurrent in $RGList) 
		{
			if ($RGCurrent.name -ne $null) { 
				$RGCurrentName = $RGCurrent.name 
				if ($RGCurrentName -is [array]) {
					$RGCurrentName = $RGCurrentName[0]
				}

				echo ""
				echo ""
				echo "There appears to be an existing TechIDMangaer.EntraID installation. "
				$replaceTIM = Read-Host -Prompt ("Do you want to update the current TechIDAgent.EntraID installation named " + $RGCurrentName + "? (Yes|No)")
				if ($replaceTIM -eq "Yes" -or $replaceTIM -eq "yes" -or $replaceTIM -eq "Y" -or $replaceTIM -eq "y") {
					$prefix_len = "techidmgr".Length
					$suffix_pos = $RGCurrentName.IndexOf("rg");
					$suffix = $RGCurrentName.Substring($prefix_len, $suffix_pos - $prefix_len)	
					$NewInstall = 'N'
					$found_something_to_do = 'Y'
					break
				} elseif ($replaceTIM -eq "No" -or $replaceTIM -eq "no" -or $replaceTIM -eq "N" -or $replaceTIM -eq "n"){
					$replaceTIM = Read-Host -Prompt "Do you want to install a new instance of TechIDAgent.EntraID in this subscription? (Yes|No)"
					if ($replaceTIM -eq "Yes" -or $replaceTIM -eq "yes" -or $replaceTIM -eq "Y" -or $replaceTIM -eq "y") {
						$NewInstall = 'Y'
						$found_something_to_do = 'Y'
						$DomainGuid = [guid]::NewGuid().Guid
						break
					} 		
				}	
			}
			if ($found_something_to_do -eq "Y") {
				break
			}
		}
		if ($found_something_to_do -eq "Y") {
			break
		}
	}

} catch { 
	Write-Error 'Could not check for existing install of TechIDAgent.EntraID from RuffianSoftware.com' -ErrorAction Stop
}

if ($found_something_to_do -ne "Y") {
	$newTIM = Read-Host -Prompt "Do you want to install a new instance of TechIDAgent.EntraID? (Yes|No)"
	if ($newTIM -eq "Yes" -or $newTIM -eq "yes" -or $newTIM -eq "Y" -or $newTIM -eq "y") {
		$NewInstall = 'Y'
	} else {
		Write-Error 'Not updating and not installing a new instance, so exiting.' -ErrorAction Stop
	}
}



$baseTIM = Read-Host -Prompt "Is this the Base installation of TechIDAgent.EntraID? (Yes|No)"
$BaseInstall = 'N'
if ($baseTIM -eq "Yes" -or $baseTIM -eq "yes" -or $baseTIM -eq "Y" -or $baseTIM -eq "y") {
	$BaseInstall = 'Y'
} 


if ($NewInstall -eq "Y") {

	try {

		$sub_count = (az account list | ConvertFrom-Json).Count
		if ($sub_count -eq 0) {
			Write-Error 'There are no subscriptions with this tenant, a subscription capable of running a function on a consumption plan is needed (pay-as-you-go)' -ErrorAction Stop
		}
		if ($sub_count -ne 1) {
			$set_sub = Read-Host -Prompt "You might have already selected a subscription during login, because there is more than one subscriptions in this tenant. Do you want to specify which subscription to use now? (Yes|No) "
		 
			if ($set_sub -eq "Yes" -or $set_sub -eq "yes" -or $set_sub -eq "Y" -or $set_sub -eq "y") {
				$subscription_to_use = Read-Host -Prompt "What Subscription ""id"" or Subscription ""name"" (listed above) do you want to use? "

				# FIXME check for a valid subscription id or name 

				az account set --subscription $subscription_to_use

			} elseif ($set_sub -eq "No" -or $set_sub -eq "no" -or $set_sub -eq "N" -or $set_sub -eq "n"){

			} else {
				Write-Error 'You did not enter Yes or No' -ErrorAction Stop
			}
		}

	} catch { 
		Write-Error 'Could not check subscriptions' -ErrorAction Stop
	}
	
	# Define the file path to save the string
	$ClientGuidFile = "$PathRSNoVersion\ClientGuid.txt"

	# Check if the file exists and read the saved string
	if (Test-Path $ClientGuidFile) {
		$defaultString = Get-Content $ClientGuidFile
	} else {
		$defaultString = ""
	}

	# this is the same client guid from the your TechIDAgent TechClient and all other agent installs.
	$ClientGuid = Read-Host -Prompt "Enter your ClientGuid (default: $defaultString)"

	# If the user input is empty, use the default string
	if ($ClientGuid -eq "") {
		$ClientGuid = $defaultString
	}

	# Save the user input to the file
	$ClientGuid | Out-File -FilePath  $ClientGuidFile -Force

	if ($BaseInstall -ne 'Y') {
		# these are for searching and should be set to something useful
		$RMMName = Read-Host -Prompt "Enter the RMM Name for this tenant (you can change this later)"
		$FriendlyName = Read-Host -Prompt "Enter the Friendly Name for this tenant (you can change this later)"

		# set this to the domain you want accounts created with for Pure Azure AD deployments
		# set this to the domain that accounts are created with by ADConnect for Hybrid deployments
		# leaving this as MyDomain.onmicrosoft.com will leave it unset and TechIDAgent will try to figure it out
		$DomainName = Read-Host -Prompt "Enter the DomainName for this tenant (you can leave this blank to let TechIDAgent pick one of the .onmicrosoft.com/us domains)"

		$UsageLocation = Read-Host -Prompt "Enter the UsageLocation for this tenant (you can change this later)"

		# for a Hybrid deployment (or a deployement where any account was created with ADConnect) this needs to be set
		# if it is left as 'Yes|No' then TechIDAgent will try to figure out what to do based on the existance of synced accounts. 
		$Hybrid = Read-Host -Prompt "Is this a Hybrid Setup (Yes|No)"

		# set this to the DomainGuid of the Domain Controller where DomainService is running and accounts are getting synced from.
		# if this is NOT a Hybrid setup the value can be left as all x's
		$HybridDomainGuid = 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'

		if ($Hybrid -eq "Yes" -or $Hybrid -eq "yes" -or $Hybrid -eq "Y" -or $Hybrid -eq "y") {
			$Hybrid = "Yes"
			$HybridDomainGuid = Read-Host -Prompt "Enter the DomainGuid of the On-Prem Domain Controller being synced"
		} elseif ($Hybrid -eq "No" -or $Hybrid -eq "no" -or $Hybrid -eq "N" -or $Hybrid -eq "n"){
			$Hybrid = "No"
		} else {
			Write-Error 'You did not enter Yes or No' -ErrorAction Stop
		}
	

		if ($Hybrid -eq  "No") 
		{
			$UserName = Read-Host -Prompt "Enter the UserName for this tenant (you can leave this blank to use the default {user}.{company})"
		
			$DisplayName = Read-Host -Prompt "Enter the DisplayName for this tenant (you can leave this blank to use the default {first} {last})"
		}

		# ask if this is a JIT setup
		$JustInTime = Read-Host -Prompt "Should TechIDAgent make these accounts as JustInTime (Yes|No)"
		if ($JustInTime -eq "Yes" -or $JustInTime -eq "yes" -or $JustInTime -eq "Y" -or $JustInTime -eq "y") {
			$JustInTime = "Yes"		
		} elseif ($JustInTime -eq "No" -or $JustInTime -eq "no" -or $JustInTime -eq "N" -or $JustInTime -eq "n"){
			$JustInTime = "No"
		} else {
			Write-Error 'You did not enter Yes or No' -ErrorAction Stop
		}

		# check all the variable to make sure they are set. 
		if ($ClientGuid -eq 'yyyyyyyy-yyyy-yyyy-yyyy-yyyyyyyyyyyy' -or $ClientGuid -eq '' -or $ClientGuid -eq $null ) {
			Write-Error '$ClientGuid needs to be set' -ErrorAction Stop
		}

		if ($Hybrid -eq 'Yes') {
			if ($HybridDomainGuid -eq 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx' -or $HybridDomainGuid -eq '') {
				Write-Error '$HybridDomainGuid must be set if Hybrid is Yes' -ErrorAction Stop
			}
		}
	}

	$regiondefault = 'eastus'
	if ($government -eq 'Y') {
		$regiondefault = 'usgovvirginia'
	}

	$tidregion = Read-Host -Prompt "What region do you want this function to run in? (you can leave this blank to use $regiondefault)"
	if ($tidregion -eq "")
	{
		$tidregion = $regiondefault
	}

	# Verify the region is valid	
	$sub_count = (az account list-locations --query "[?name == '$tidregion' && metadata.regionType != 'Logical' ]" | ConvertFrom-Json).Count
	if ($sub_count -eq 0) {
		az account list-locations --query "[?metadata.regionType != 'Logical' ]" | ConvertFrom-Json
		Write-Error "Region $tidregion is not valid. Try again." 

		$tidregion = Read-Host -Prompt "What region do you want this function to run in?"

		$sub_count = (az account list-locations --query "[?name == '$tidregion' && metadata.regionType != 'Logical' ]" | ConvertFrom-Json).Count
		if ($sub_count -eq 0) {
			Write-Error "Region $tidregion is not valid. Try again and use one from the above list." -ErrorAction Stop
		}
	}
}

echo "Pausing before download and install"
Start-Sleep 3

$VERSION = "6.147"

$SourceFile = "https://techidmanager.com/wp-content/uploads/2025/10/TechIDAgent.EntraID_Base_6.147.zip"
$DestFile = "$PathRSVersion\TechIDAgent.EntraID_Base_6.147.zip"

try {
	# make and change directory to the location of the downloaded zipfile TechIDAgent.EntraID_?.??.zip	
	if (Test-Path "$DestFile") 
	{
		# if the file already exists AND the hash is right, just use it.
		$hash = (Get-FileHash -Path "$DestFile").Hash
		if ($hash -ne "7EF78BEA3273AB9A711D12C07C4B7D1AA611C2CBE515BA90566D068D9E38437D")
		{		
			echo "$DestFile -> $hash"
			echo "Hash of existing downloading installation file doesn't match expected value, deleting."
			Remove-Item "$DestFile"
			# seems like we need to delay before overwritting a file we just deleted? 
			Start-Sleep 1
		}		
	}
	if (-not(Test-Path "$DestFile")) 
	{
		echo "Downloading TechIDAgent.EntraID installation file to $DestFile."
		Invoke-WebRequest -Uri $SourceFile  -OutFile "$DestFile"
	}
} catch {
	Write-Error 'Could not download TechIDAgent.EntraID from RuffianSoftware.com' -ErrorAction Stop
}

$hash = (Get-FileHash -Path "$DestFile").Hash
if ($hash -ne "7EF78BEA3273AB9A711D12C07C4B7D1AA611C2CBE515BA90566D068D9E38437D")
{
	echo "$DestFile -> $hash"
	Write-Error 'Hash of downloaded file does not match expected value.' -ErrorAction Stop
}

try {

	$RGname = 'techidmgr' + $suffix + 'rg' 
	$APname = 'techidmgr' + $suffix + 'fa'
	$SAname = 'techidmgr' + $suffix + 'sa'
	$SPname = 'techidmgr' + $suffix + 'sp'

	
	if ($NewInstall -eq "Y") {

		#needed on all these commands to limit output to a readable level
		# | out-null
		# or 
		# | Out-File -Append -FilePath  EntraID.Install.Log



		# This might not be required, but it doesn't always happen automatacially. 
		echo "Registering storage Provider"
		az provider register --namespace Microsoft.Storage --wait | Out-File -Append -FilePath  EntraID.Install.Log
		# And there seems to be a delay between when it says it is done and when it is usable. 
		Start-Sleep 3

		echo "Registering web Provider"
		az provider register --namespace Microsoft.Web --wait | Out-File -Append -FilePath  EntraID.Install.Log
		# And there seems to be a delay between when it says it is done and when it is usable. 
		Start-Sleep 3

		# Create a resource resourceGroupName
		echo "Creating resource group with name $RGname"
		az group create --name $RGname --location $tidregion  | Out-File -Append -FilePath  EntraID.Install.Log
		# all these delays are because often these az calls will return and say they are done before they are actually done? It appears. 
		Start-Sleep 3	

		# Create an azure storage account
		echo "Creating storage account with name $SAname"
		az storage account create --name $SAname --location $tidregion --resource-group $RGname --sku Standard_LRS  | Out-File -Append -FilePath  EntraID.Install.Log
		Start-Sleep 3

		# Create a Function App
		echo "Creating function app plan with name $SPname "
		az functionapp plan create --name $SPname --resource-group $RGname --location $tidregion --sku F1 --debug | Out-File -Append -FilePath  EntraID.Install.Log		
		echo "Done function app plan with name $SPname " | Out-File -Append -FilePath  EntraID.Install.Log
		Start-Sleep 3

		echo "Creating function app with name $APname "
		# "--assign-identity [system]" had to be removed from the following line 11/26/2024
		az functionapp create --name $APname --storage-account $SAname --resource-group $RGname --functions-version 4 --runtime dotnet-isolated --runtime-version 8  --consumption-plan-location $tidregion --disable-app-insights --assign-identity [system] | Out-File -Append -FilePath  EntraID.Install.Log
		Start-Sleep 3

		# Set the configuration variables.
		if ($government -eq "Y") {
			echo "setting configuration variable TechIDManager.AzureGraphCloud"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.AzureGraphCloud=usgov" | Out-File -Append -FilePath  EntraID.Install.Log
		}
		echo "setting configuration variable TechIDManager.ClientGuid"
		az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.ClientGuid=$ClientGuid" | Out-File -Append -FilePath  EntraID.Install.Log

		
		if ($BaseInstall -ne 'Y') {
			echo "setting configuration variable TechIDManager.RMMName"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.RMMName=$RMMName" | Out-File -Append -FilePath  EntraID.Install.Log
			echo "setting configuration variable TechIDManager.FriendlyName"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.FriendlyName=$FriendlyName" | Out-File -Append -FilePath  EntraID.Install.Log
			echo "setting configuration variable TechIDManager.HourToRun"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.HourToRun=5" | Out-File -Append -FilePath  EntraID.Install.Log

			echo "setting configuration variable TechIDManager.Hybrid"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.Hybrid=$Hybrid" | Out-File -Append -FilePath  EntraID.Install.Log
			echo "setting configuration variable TechIDManager.HybridDomainGuid"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.HybridDomainGuid=$HybridDomainGuid" | Out-File -Append -FilePath  EntraID.Install.Log

			echo "setting configuration variable TechIDManager.JustInTime"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.JustInTime=$JustInTime" | Out-File -Append -FilePath  EntraID.Install.Log

			if ($DomainGuid -ne "") {
				echo "setting configuration variable TechIDManager.DomainGuid"
				az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.DomainGuid=$DomainGuid" | Out-File -Append -FilePath  EntraID.Install.Log
			}

			#setting the domainname is not required.
			echo "setting configuration variable TechIDManager.DomainName"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.DomainName=$DomainName" | Out-File -Append -FilePath  EntraID.Install.Log
			echo "setting configuration variable TechIDManager.UserName"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.UserName=$UserName" | Out-File -Append -FilePath  EntraID.Install.Log
			echo "setting configuration variable TechIDManager.DisplayName"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.DisplayName=$DisplayName" | Out-File -Append -FilePath  EntraID.Install.Log
			echo "setting configuration variable TechIDManager.UsageLocation"
			az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.UsageLocation=$UsageLocation" | Out-File -Append -FilePath  EntraID.Install.Log
		}
	} 

	echo "setting configuration variable TechIDManager.Primary"
	az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.Primary=$BaseInstall" | Out-File -Append -FilePath  EntraID.Install.Log
	echo "setting configuration variable TechIDManager.Base"
	az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.Base=$BaseInstall" | Out-File -Append -FilePath  EntraID.Install.Log

	# set the functionapp extensions version to 4 because that is what we are using. 	
	echo "setting configuration variable for FUNCTIONS_EXTENSION_VERSION=~4"	
	az functionapp config appsettings set --name $APname --resource-group $RGname --settings FUNCTIONS_EXTENSION_VERSION=~4 | Out-File -Append -FilePath  EntraID.Install.Log
	
	# we might be updating an install, which might be a different version of .net and function version....so we have to set that 
	echo "setting function app runtime and .net version with name $APname "
	az functionapp config appsettings set --name $APname --resource-group $RGname --settings FUNCTIONS_WORKER_RUNTIME=dotnet-isolated DOTNET_ISOLATED_VERSION=8.0 | Out-File -Append -FilePath  EntraID.Install.Log
	

	# FIXME look at using a vault to store this keys
	$currentSettings = az functionapp config appsettings list --name $APname --resource-group $RGname | ConvertFrom-Json
	$privateKeySetting = $currentSettings | Where-Object { $_.name -eq "TechIDManager.DomainPrivateKey" }

	if (-not $privateKeySetting) {
		# Generate a new 2048-bit RSA key
		$RSA = [System.Security.Cryptography.RSA]::Create(2048)

		# Export the RSA key to XML
		$keyXML = $RSA.ToXmlString($true)
		$keyBase64 = [Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($keyXML))

		# Set the environment variable to the XML for the RSA key
		az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.DomainPrivateKey=$keyBase64" | Out-File -Append -FilePath  EntraID.Install.Log

		# Output the new key
		Write-Output "New RSA key generated and set in the environment variable"
	} else {
		# Output the existing key
		Write-Output "Existing RSA key found in the environment variable"
	}
	

	# upload the code file for the function to the latest zipfile
	echo "Deploying function code for version $VERSION"
	az functionapp deployment source config-zip --name $APname --resource-group $RGname --src "$DestFile" | Out-File -Append -FilePath  EntraID.Install.Log
	Start-Sleep 3

	echo "Finding function URL"
	$TriggerName = "Run"
	$DefaultHostName = (az webapp config hostname list --resource-group $RGName --webapp-name $APName | ConvertFrom-Json).name
	$FunctionKey = (az functionapp function keys list --function-name Run --name $APName --resource-group $RGName  | ConvertFrom-Json).default
	$FunctionURL = az functionapp function show --function-name $TriggerName --name $APname --resource-group $RGname --query "invokeUrlTemplate" --output tsv
	$INVOKE_FUNCTION = -join ( $FunctionURL, "?code=", $FunctionKey )

	echo "Setting function URL"
	az functionapp config appsettings set --name $APname --resource-group $RGname --settings "TechIDManager.CallURL=$INVOKE_FUNCTION" | Out-File -Append -FilePath  EntraID.Install.Log

	#a base install does not need Global admin permissions. It's permissions come from the tokens givens to secondary installs. 
	if ($BaseInstall -eq 'N') {

		# check for the function not having Global administrator and then add it. 
	
		$FunctionID = (az ad sp list --display-name $APName | ConvertFrom-Json).objectId
		if ($FunctionID -eq $null -or $FunctionID -eq '' ) {
			echo 'FunctionID blank so using id instead of objectId'
			$FunctionID = (az ad sp list --display-name $APName | ConvertFrom-Json).id
		}

		echo ('FunctionID = ' + $FunctionID)

		$roleName = 'Global Administrator'	
	
		# Obtain an access token for Microsoft Graph API
		$token = az account get-access-token --resource "$basegraph" --query accessToken --output tsv
			
		# Get the Global Administrator role ID
		$directoryRolesResponse  = az rest --method get --uri "$basegraph/v1.0/directoryRoles?`$filter=displayName eq '$roleName'" --headers "Authorization=Bearer $token" --output json
		$roleDefinitionId = ($directoryRolesResponse | ConvertFrom-Json).value | Where-Object { $_.displayName -eq $roleName } | Select-Object -ExpandProperty Id
		$roleTemplateId = ($directoryRolesResponse | ConvertFrom-Json).value | Where-Object { $_.displayName -eq $roleName } | Select-Object -ExpandProperty roleTemplateId

		if ([string]::IsNullOrEmpty($roleDefinitionId) -or [string]::IsNullOrEmpty($roleTemplateId)) {
			Write-Error "$roleName role not found"
		}

		#testing this to check role assignments. 
		$roleAssignmentsResponse = az rest --method get --uri "$basegraph/v1.0/roleManagement/directory/roleAssignments?`$filter=principalId eq '$FunctionID'" --headers "Authorization=Bearer $token" --output json
		$roleAssignments = ($roleAssignmentsResponse | ConvertFrom-Json).value
		$alreadyAssigned = $roleAssignments | Where-Object { $_.principalId -eq $FunctionID -and $_.roleDefinitionId -eq $roleTemplateId }


		if ($alreadyAssigned)				
		{
			echo "$roleName already applied to function"

		}	
		else	
		{
			# grant the function app the role it needs to create/disable users and set passwords. Only the "Global Administrator" role is allowed to do this. 
	
			echo "Applying $roleName to function"
	
			# Assign the Global Administrator role to the Service Principal
			$body = @{
				principalId = $FunctionID
				roleDefinitionId = $roleDefinitionId
				directoryScopeId = "/"
			} | ConvertTo-Json

			# Make sure to format the JSON payload properly
			$body = $body -replace """", "\"""
			$body = $body -replace "`r`n", "" -replace "`n", "" -replace "`r", ""
		
			$response = az rest --method post --uri "$basegraph/v1.0/roleManagement/directory/roleAssignments" --headers "Authorization=Bearer $token" "Content-Type=application/json" --body $body

			echo "Global Administrator role assigned to Service Principal for TechIDManager $APId"
			# the above permission setting line will successed instantly, but the permission will not really be applied for a while. 
			# 30 seconds was not long enough, so we are trying 60.
			echo "Waiting for role to be applied to function..."
			Start-Sleep 60

		}
	}

	echo "getting webpage " + $INVOKE_FUNCTION 
	(Invoke-WebRequest -Uri $INVOKE_FUNCTION).Content	

	echo ""
	echo ""
	echo "Check output above for errors. If there are errors contact support@RuffianSoftware.com "
	echo "If there are no errors you are done. "


} catch {
	$message = $_
	Write-Error "something went wrong contact support@RuffianSoftware.com $message" -ErrorAction Stop
}

# SIG # Begin signature block
# MIIt/wYJKoZIhvcNAQcCoIIt8DCCLewCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUYH5X2S3u5J7KDXcxLquC7nvo
# 5m6ggidZMIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0B
# AQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz
# 7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS
# 5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
# bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfI
# SKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jH
# trHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14
# Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2
# h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt
# 6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPR
# iQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ER
# ElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4K
# Jpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAd
# BgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SS
# y4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAC
# hjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURS
# b290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRV
# HSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyh
# hyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO
# 0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo
# 8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++h
# UD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5x
# aiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMIIFkDCCA3ig
# AwIBAgIQBZsbV56OITLiOQe9p3d1XDANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQG
# EwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNl
# cnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMTMw
# ODAxMTIwMDAwWhcNMzgwMTE1MTIwMDAwWjBiMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYD
# VQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEppz1Y
# q3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lX
# FllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYyktzuxe
# TsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbu
# yntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I
# 9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmg
# Z92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse
# 5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKy
# Ebe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwh
# HbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98THU/
# Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3lGwID
# AQABo0IwQDAPBgNVHRMBAf8EBTADAQH/MA4GA1UdDwEB/wQEAwIBhjAdBgNVHQ4E
# FgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wDQYJKoZIhvcNAQEMBQADggIBALth2X2p
# bL4XxJEbw6GiAI3jZGgPVs93rnD5/ZpKmbnJeFwMDF/k5hQpVgs2SV1EY+CtnJYY
# ZhsjDT156W1r1lT40jzBQ0CuHVD1UvyQO7uYmWlrx8GnqGikJ9yd+SeuMIW59mdN
# Oj6PWTkiU0TryF0Dyu1Qen1iIQqAyHNm0aAFYF/opbSnr6j3bTWcfFqK1qI4mfN4
# i/RN0iAL3gTujJtHgXINwBQy7zBZLq7gcfJW5GqXb5JQbZaNaHqasjYUegbyJLkJ
# EVDXCLG4iXqEI2FCKeWjzaIgQdfRnGTZ6iahixTXTBmyUEFxPT9NcCOGDErcgdLM
# MpSEDQgJlxxPwO5rIHQw0uA5NBCFIRUBCOhVMt5xSdkoF1BN5r5N0XWs0Mr7QbhD
# parTwwVETyw2m+L64kW4I1NsBm9nVX9GtUw/bihaeSbSpKhil9Ie4u1Ki7wb/UdK
# Dd9nZn6yW0HQO+T0O/QEY+nvwlQAUaCKKsnOeMzV6ocEGLPOr0mIr/OSmbaz5mEP
# 0oUA51Aa5BuVnRmhuZyxm7EAHu/QD09CbMkKvO5D+jpxpchNJqU1/YldvIViHTLS
# oCtU7ZpXwdv6EM8Zt4tKG48BtieVU+i2iW1bvGjUI+iLUaJW+fCmgKDWHrO8Dw9T
# dSmq6hN35N6MgSGtBxBHEa2HPQfRdbzP82Z+MIIGsDCCBJigAwIBAgIQCK1AsmDS
# nEyfXs2pvZOu2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYD
# VQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAwWhcN
# MzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQs
# IEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
# IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8A
# MIICCgKCAgEA1bQvQtAorXi3XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXr
# NCQH4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAkZLON4gh9NH1MgFcS
# a0OamfLFOx/y78tHWhOmTLMBICXzENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4
# RS6hnyzhGMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRXXegYE2XFf7JP
# hSxIpFaENdb5LpyqABXRN/4aBpTCfMjqGzLmysL0p6MDDnSlrzm2q2AS4+jWufcx
# 4dyt5Big2MEjR0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJawv9qYFSL
# ScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tvk2E0XLyTRSiDNipmKF+wc86LJiUG
# soPUXPYVGUztYuBeM/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eRGv7bUKJG
# yGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2FKZbS110YU0/EpF23r9Yy3IQ
# KUHw1cVtJnZoEUETWJrcJisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa
# 6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27IwyCQLMbDwMVhECAwEAAaOC
# AVkwggFVMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7
# /mF7CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9PMA4GA1Ud
# DwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkw
# JAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggrBgEFBQcw
# AoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJv
# b3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwHAYDVR0gBBUwEzAHBgVngQwB
# AzAIBgZngQwBBAEwDQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6mdNW4AIa
# pfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6jfCbVN7w6XUhtldU/SfQnuxaB
# RVD9nL22heB2fjdxyyL3WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzYgBoRGRjN
# YZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGVWEjVGv7XJz/9kNF2ht0c
# sGBc8w2o7uCJob054ThO2m67Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3Z
# XQIUH0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m6Ri+kAewQ3+ViCCC
# cPDMyu/9KTVcH4k4Vfc3iosJocsL6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJ
# zy6kdJKt2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFSi0HASIRLlk2r
# REDjjfAVKM7t8RhWByovEMQMCGQ8M4+uKIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm
# 5yg/rBKupS8ibEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhRILutG4UI
# 4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+FSCH5Vzu0nAPthkX0tGFuv2jiJmCG
# 6sivqf6UHedjGzqGVnhOMIIGtDCCBJygAwIBAgIQDcesVwX/IZkuQEMiDDpJhjAN
# BgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQg
# SW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2Vy
# dCBUcnVzdGVkIFJvb3QgRzQwHhcNMjUwNTA3MDAwMDAwWhcNMzgwMTE0MjM1OTU5
# WjBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNV
# BAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0YW1waW5nIFJTQTQwOTYgU0hB
# MjU2IDIwMjUgQ0ExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAtHgx
# 0wqYQXK+PEbAHKx126NGaHS0URedTa2NDZS1mZaDLFTtQ2oRjzUXMmxCqvkbsDpz
# 4aH+qbxeLho8I6jY3xL1IusLopuW2qftJYJaDNs1+JH7Z+QdSKWM06qchUP+AbdJ
# gMQB3h2DZ0Mal5kYp77jYMVQXSZH++0trj6Ao+xh/AS7sQRuQL37QXbDhAktVJMQ
# bzIBHYJBYgzWIjk8eDrYhXDEpKk7RdoX0M980EpLtlrNyHw0Xm+nt5pnYJU3Gmq6
# bNMI1I7Gb5IBZK4ivbVCiZv7PNBYqHEpNVWC2ZQ8BbfnFRQVESYOszFI2Wv82wnJ
# RfN20VRS3hpLgIR4hjzL0hpoYGk81coWJ+KdPvMvaB0WkE/2qHxJ0ucS638ZxqU1
# 4lDnki7CcoKCz6eum5A19WZQHkqUJfdkDjHkccpL6uoG8pbF0LJAQQZxst7VvwDD
# jAmSFTUms+wV/FbWBqi7fTJnjq3hj0XbQcd8hjj/q8d6ylgxCZSKi17yVp2NL+cn
# T6Toy+rN+nM8M7LnLqCrO2JP3oW//1sfuZDKiDEb1AQ8es9Xr/u6bDTnYCTKIsDq
# 1BtmXUqEG1NqzJKS4kOmxkYp2WyODi7vQTCBZtVFJfVZ3j7OgWmnhFr4yUozZtqg
# PrHRVHhGNKlYzyjlroPxul+bgIspzOwbtmsgY1MCAwEAAaOCAV0wggFZMBIGA1Ud
# EwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFO9vU0rp5AZ8esrikFb2L9RJ7MtOMB8G
# A1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9PMA4GA1UdDwEB/wQEAwIBhjAT
# BgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUHMAGG
# GGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggrBgEFBQcwAoY1aHR0cDovL2Nh
# Y2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYD
# VR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9
# bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQAXzvsWgBz+Bz0RdnEwvb4LyLU0pn/N0IfF
# iBowf0/Dm1wGc/Do7oVMY2mhXZXjDNJQa8j00DNqhCT3t+s8G0iP5kvN2n7Jd2E4
# /iEIUBO41P5F448rSYJ59Ib61eoalhnd6ywFLerycvZTAz40y8S4F3/a+Z1jEMK/
# DMm/axFSgoR8n6c3nuZB9BfBwAQYK9FHaoq2e26MHvVY9gCDA/JYsq7pGdogP8HR
# trYfctSLANEBfHU16r3J05qX3kId+ZOczgj5kjatVB+NdADVZKON/gnZruMvNYY2
# o1f4MXRJDMdTSlOLh0HCn2cQLwQCqjFbqrXuvTPSegOOzr4EWj7PtspIHBldNE2K
# 9i697cvaiIo2p61Ed2p8xMJb82Yosn0z4y25xUbI7GIN/TpVfHIqQ6Ku/qjTY6hc
# 3hsXMrS+U0yy+GWqAXam4ToWd2UQ1KYT70kZjE4YtL8Pbzg0c1ugMZyZZd/BdHLi
# Ru7hAWE6bTEm4XYRkA6Tl4KSFLFk43esaUeqGkH/wyW4N7OigizwJWeukcyIPbAv
# jSabnf7+Pu0VrFgoiovRDiyx3zEdmcif/sYQsfch28bZeUz2rtY/9TCA6TD8dC3J
# E3rYkrhLULy7Dc90G6e8BlqmyIjlgp2+VqsS9/wQD7yFylIz0scmbKvFoW2jNrbM
# 1pD2T7m3XDCCBu0wggTVoAMCAQICEAqA7xhLjfEFgtHEdqeVdGgwDQYJKoZIhvcN
# AQELBQAwaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMUEw
# PwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVTdGFtcGluZyBSU0E0MDk2
# IFNIQTI1NiAyMDI1IENBMTAeFw0yNTA2MDQwMDAwMDBaFw0zNjA5MDMyMzU5NTla
# MGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UE
# AxMyRGlnaUNlcnQgU0hBMjU2IFJTQTQwOTYgVGltZXN0YW1wIFJlc3BvbmRlciAy
# MDI1IDEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDQRqwtEsae0Oqu
# YFazK1e6b1H/hnAKAd/KN8wZQjBjMqiZ3xTWcfsLwOvRxUwXcGx8AUjni6bz52fG
# Tfr6PHRNv6T7zsf1Y/E3IU8kgNkeECqVQ+3bzWYesFtkepErvUSbf+EIYLkrLKd6
# qJnuzK8Vcn0DvbDMemQFoxQ2Dsw4vEjoT1FpS54dNApZfKY61HAldytxNM89PZXU
# P/5wWWURK+IfxiOg8W9lKMqzdIo7VA1R0V3Zp3DjjANwqAf4lEkTlCDQ0/fKJLKL
# kzGBTpx6EYevvOi7XOc4zyh1uSqgr6UnbksIcFJqLbkIXIPbcNmA98Oskkkrvt6l
# PAw/p4oDSRZreiwB7x9ykrjS6GS3NR39iTTFS+ENTqW8m6THuOmHHjQNC3zbJ6nJ
# 6SXiLSvw4Smz8U07hqF+8CTXaETkVWz0dVVZw7knh1WZXOLHgDvundrAtuvz0D3T
# +dYaNcwafsVCGZKUhQPL1naFKBy1p6llN3QgshRta6Eq4B40h5avMcpi54wm0i2e
# PZD5pPIssoszQyF4//3DoK2O65Uck5Wggn8O2klETsJ7u8xEehGifgJYi+6I03Uu
# T1j7FnrqVrOzaQoVJOeeStPeldYRNMmSF3voIgMFtNGh86w3ISHNm0IaadCKCkUe
# 2LnwJKa8TIlwCUNVwppwn4D3/Pt5pwIDAQABo4IBlTCCAZEwDAYDVR0TAQH/BAIw
# ADAdBgNVHQ4EFgQU5Dv88jHt/f3X85FxYxlQQ89hjOgwHwYDVR0jBBgwFoAU729T
# SunkBnx6yuKQVvYv1Ensy04wDgYDVR0PAQH/BAQDAgeAMBYGA1UdJQEB/wQMMAoG
# CCsGAQUFBwMIMIGVBggrBgEFBQcBAQSBiDCBhTAkBggrBgEFBQcwAYYYaHR0cDov
# L29jc3AuZGlnaWNlcnQuY29tMF0GCCsGAQUFBzAChlFodHRwOi8vY2FjZXJ0cy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRUaW1lU3RhbXBpbmdSU0E0MDk2
# U0hBMjU2MjAyNUNBMS5jcnQwXwYDVR0fBFgwVjBUoFKgUIZOaHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0VGltZVN0YW1waW5nUlNBNDA5
# NlNIQTI1NjIwMjVDQTEuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG
# /WwHATANBgkqhkiG9w0BAQsFAAOCAgEAZSqt8RwnBLmuYEHs0QhEnmNAciH45PYi
# T9s1i6UKtW+FERp8FgXRGQ/YAavXzWjZhY+hIfP2JkQ38U+wtJPBVBajYfrbIYG+
# Dui4I4PCvHpQuPqFgqp1PzC/ZRX4pvP/ciZmUnthfAEP1HShTrY+2DE5qjzvZs7J
# IIgt0GCFD9ktx0LxxtRQ7vllKluHWiKk6FxRPyUPxAAYH2Vy1lNM4kzekd8oEARz
# FAWgeW3az2xejEWLNN4eKGxDJ8WDl/FQUSntbjZ80FU3i54tpx5F/0Kr15zW/mJA
# xZMVBrTE2oi0fcI8VMbtoRAmaaslNXdCG1+lqvP4FbrQ6IwSBXkZagHLhFU9HCrG
# /syTRLLhAezu/3Lr00GrJzPQFnCEH1Y58678IgmfORBPC1JKkYaEt2OdDh4GmO0/
# 5cHelAK2/gTlQJINqDr6JfwyYHXSd+V08X1JUPvB4ILfJdmL+66Gp3CSBXG6IwXM
# ZUXBhtCyIaehr0XkBoDIGMUG1dUtwq1qmcwbdUfcSYCn+OwncVUXf53VJUNOaMWM
# ts0VlRYxe5nK+At+DI96HAlXHAL5SlfYxJ7La54i71McVWRP66bW+yERNpbJCjyC
# YG2j+bdpxo/1Cy4uPcU3AWVPGrbn5PhDBf3Froguzzhk++ami+r3Qrx5bIbY3TVz
# giFI7Gq3zWcwggfTMIIFu6ADAgECAhALBiMNC9gx0k3bf3Vr7kCeMA0GCSqGSIb3
# DQEBDAUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFB
# MD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcgUlNBNDA5
# NiBTSEEzODQgMjAyMSBDQTEwHhcNMjUxMTA3MDAwMDAwWhcNMjcxMTE2MjM1OTU5
# WjCB2zETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgECEwhEZWxh
# d2FyZTEdMBsGA1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xEDAOBgNVBAUTBzcz
# MjA3ODQxCzAJBgNVBAYTAlVTMRcwFQYDVQQIEw5Tb3V0aCBDYXJvbGluYTEQMA4G
# A1UEBxMHTWF1bGRpbjEfMB0GA1UEChMWUnVmZmlhbiBTb2Z0d2FyZSwgSW5jLjEf
# MB0GA1UEAxMWUnVmZmlhbiBTb2Z0d2FyZSwgSW5jLjCCAiIwDQYJKoZIhvcNAQEB
# BQADggIPADCCAgoCggIBAKcm6hmeuV3+KtqdHLIMBmogsWHGHYK73PuZpTxOOol8
# HQkacVGg0wYIq34axHaXkcsvI1jMwzuNUvypWqHaONfBK2FaFdAwTsLRmlhvFxbL
# p0Zw1wQY+5jxjyzogiIqVEQbZuxK7UHTKi6sSLUC8gLdfYAK4GDBRPSAevFAa/Ln
# S8A0LKf06I8wuScHGcJ+vi13U+zfxC03bWI2Ifl2LcElVPNOQCiaDbOpYtqJiXjz
# w8po7VeifRuarNccn1htfbFSlllbJvh0hXd+GsGGZxzTs/l1/rHMj2RBl+qCq2/A
# HvvlT2AiNj5jm7fbB/TAIyKeocPrJbk2LJ0glxKFj9LrOKgYKBjeVpyGT5XjoJ3B
# yXFH9Im5KuXDOxTFTXOfyOBaRksL9Q3QUotA0wQxTgP+Ws+W72kMsXZeDvKWUGIp
# O//mptqfXhW+fxzKAADthu504Q/UEwlPlL5usuW2gl9FRQ7J8t2J0uejvAORm46e
# 5ppN1CwjSOPcaXwJNhmHb7suIxjGO5Kr74xxDDmKfCPfObF9gAF5W+uBifL2/bNb
# Gp2ctIJIljzZEEVgBf0Go7B0nlsxcjj9IVfH3YzJ0X6WiZAdGQ01Cyy6kuiZ4JO9
# jAsR00supRpPpLAWN7pO3VvsdwXPinNQ4nWw5PauzDXYuBnrOWC9xpgE8AQJt3jB
# AgMBAAGjggICMIIB/jAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAd
# BgNVHQ4EFgQUL52SdlnZVaaXrDZvDfleA1LExNkwPQYDVR0gBDYwNDAyBgVngQwB
# AzApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwDgYD
# VR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0wgaow
# U6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRH
# NENvZGVTaWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRw
# Oi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmlu
# Z1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBcBggrBgEFBQcwAoZQ
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29k
# ZVNpZ25pbmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0TBAIwADANBgkq
# hkiG9w0BAQwFAAOCAgEAp3Ra4p/3fOd07wHhYniXcVM9VGi+BRDbRUa04sgfGWl4
# /fF+68HTnXatJxfztSg9qrzgwP6OGhLZp051iYWea8mvZeMe03u+/7xInUdfdSLj
# h+VkBh5AQFKQuP+3PoYFKXOmrULHd9e3YXTl6Q9FCd0JFzkqCvbBQkp2DWDonAe4
# 8Vr/bmkcF+F1gxh8gJQJWAXaUWHt7+5zy7WyFqgqocpIkIsEwPxazbd/bzEFwC4u
# Q1zwEi8Ey0PcD4otnsmqidYSK2X0nFkCLdzeAR1wEIQ8LQV0tQVCEcdXPTuCIM1M
# BBCzGijqp5tPs+yM830iv5GWPLU9LmMec2SI/jsV9Z7fJrXoRLaXg5/soIpqwM7A
# qHtqtufuiXvoEVVUge8jmF0Sd/N0WNaW/fBTvPoRvhIBwDfQBSgguDZAOQ3w3jV0
# bFLVeUORhsDtuC1tHhq665N+WVHs9KJ9bIA1lxNtRW7Qclw19wLBb/jweOv7zPpE
# ciBEx/o63PYnBUb0DaUc/WKAHtoy44G1Wy2TA9UsSYenhH7+hTq5PMTo1qFEwJDN
# VJ0nbVAZJ3I6pzQr/l8hE34CMuUnkvctz+WY7mDO0ufIhr15agcexY9G36s6wtqx
# 2QlVRZEUz8Km460WpRxoXmDUffzxw/fATmCbFvSjTnmc52wNZyl7fKJelBJrensx
# ggYQMIIGDAIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTECEAsGIw0L2DHSTdt/dWvuQJ4wCQYFKw4D
# AhoFAKBAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMCMGCSqGSIb3DQEJBDEW
# BBSggxkqvxVy6/gxwKlddB8s9mj5ZTANBgkqhkiG9w0BAQEFAASCAgAjQaZUX4fd
# lz9pzju1HKrM4n1pdOxIuU7zX4UJPzXk+Aj+AJMKpcXM1RmvjvT3Mq36kvQiPV/9
# aWvMUn0NcH2156/5+paPtk5QD02mrMrctHb7WCPu83hwrrlm1MlZVzuuC7/yyy4p
# dR06YFELXK4olke24Vpk65Y4Z4Iyq9LQP/kQIYf2Bqd1/EC8hTNJMCaSAnUN1Vnq
# +QPVhslI6odDaN2b/ofcaL43oH8mZbFUwMTPDqQBHox9by6dWz5un0Ky0L84qTi7
# 5S2BUA9CLK0ruiyxJ4TujTIZXRPpebCSL96KsldWKT8EGhntQPjk2D89AuyGXmga
# nSxYS2VCwrgBFm+JCWj4S+NwIcAJCyZUISJUqzv/HCBqhWDRpu3CcRcbecXkim7s
# XFxfDJc33cmP//4YbMX4nueZLgH2iJ01UnfXgeY9/dpkmkIFIo1JrbZs3lVpQSWP
# Dq5vR+k3cN17uIYuDu6inrwFJuRC2/wtOIiwAMcJJrq/4M40SQRNhfNAuStZ9IRI
# xPAeiizi74Y1EeINY6De3Ek/ke19joV3Eh4HW4DuYUGS71uqKKpCrYjFWXoLAGDj
# n+A0uzUNZyr+KwPlUXbaItnFq6WhIy5qlwOcwszhboJJMxoKnINPdqpmzcMT1drC
# 7ulwO1+CuAqXiOYUsE1CGYoAwcNwxkAqEaGCAyYwggMiBgkqhkiG9w0BCQYxggMT
# MIIDDwIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5j
# LjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcgUlNB
# NDA5NiBTSEEyNTYgMjAyNSBDQTECEAqA7xhLjfEFgtHEdqeVdGgwDQYJYIZIAWUD
# BAIBBQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEP
# Fw0yNTEyMDkyMDM2MDNaMC8GCSqGSIb3DQEJBDEiBCBVt/699lXnMOen8aHaotsN
# xILGcjtaK0SN7Nc8BvSsHTANBgkqhkiG9w0BAQEFAASCAgCo3iX1rhJzSzgZM1DC
# SZptsQeS5zrsCLecm537MAbaLbpuVudTCz9WfCwzJfxJAkGewVzyTGH/aSAu0bBx
# TnYPthZ836z7QgEUU6WckJ+SoiSw2eLtjWLkQVYbf46NifKb3yDR+DmXNRU8dGJQ
# b1aY1v/wreGMOITiGTBVbNCRa/yrOIITQuCA14T0HM9yLVDP/0LuUJ42RgWQWOmF
# j6brd0VJUNE9qcefY7wrD05My8vcdXSRQm3NvT5XyOI+j6Db59RLYKq+QvshHcYz
# bCtL7FCvjoXBBPE0B1v9hvoXpQdlkXDMqpL9iyGCxOpmYMfwgkxT+qe2Pkfxf4/O
# IyKgg4G/rml0gy/bwMJ2gc2kn3hUzZ8Bl3NgZuKAETUNH5XxiPWlR1ekG1TZBCcb
# 46P9MaIVA5oow+C50cKObqFRjp81Vl7cEFA2gpwQMrX8Db8t48w4SRZpLvM+XGoS
# jckV0Z0xGfWluo9opaQob/4u6bCgq3nqm6Gtct307mwkJEnrWh2b80sfKDT3R4pR
# MGMGNaZ43Ms9Z35ySoqQ+TFha8hRs7KS0KYWpx0rw/pI/6H66O2hO4bDYm5wAmMW
# rsBV7ljns4TfN8vapvGjcEYhJ137YniwbeozTL+I3YLwQu+AvROFAYlQwxt/nx9U
# sRPkYcmxR4iyJEdHf3/MzvkIjA==
# SIG # End signature block
